;; Note: This code could be improved in various ways.  The overall
;; length of the code and the number of agent variables used is
;; excessive, and could be cut down using various techniques (lists,
;; local variables, etc.).

breeds [
    butterflies
    birds
]

butterflies-own [
    monarch?          ;; if true, the butterfly is a monarch (yucky).
                      ;;    if false, it's a viceroy (tasty).
    bird-here         ;; One of the birds
                      ;;   occupying a butterfly's patch.
]

birds-own [
    memory1           ;; Remembers color of a recently
                      ;;    eaten monarch.
    memory2           ;; Remembers color of a recently
                      ;;    eaten monarch.
    memory3           ;; Remembers color of a recently
                      ;;    eaten monarch.
    memorycount1      ;; Number of ticks that memory1 is
                      ;;    holding current color.
    memorycount2      ;; Number of ticks that memory2 is
                      ;;    holding current color.
    memorycount3      ;; Number of ticks that memory2 is
                      ;;    holding current color.
]

globals [
    ticks               ;; Time
    carry-cap-mnrchs    ;; A maximum population of monarchs
                        ;;    the world can support.
    carry-cap-vcrys     ;; A maximum population of viceroys
                        ;;    the world can support.
    carry-cap-birds     ;; A maximum population of birds
                        ;;    the world can support.
    color-range-begin   ;; The "lowest" color a butterfly
                        ;;    can be.
    color-range-end     ;; The "highest" color a butterfly
                        ;;    can be.
    reproduction-chance ;; The chance and individual has of
                        ;;    reproducing (0 - 100) *after*
                        ;;    the chance dependent on
                        ;;    carrying capacity is evaluated.
    mnrchs              ;; Number of monarchs.
    mnrch-max           ;; Highest color of the monarchs.
    mnrch-min           ;; Lowest color of the monarchs.
    mnrch-avg           ;; Average color of the monarchs.
    m-death             ;; Number of monarchs that have died.
    mnrch-death         ;; Number of monarchs that have died
                        ;;    in that turn.
    vcrys               ;; Number of viceroys.
    vcry-max            ;; Highest color of the viceroys.
    vcry-min            ;; Lowest color of the viceroys.
    vcry-avg            ;; Average color of the viceroys.
    v-death             ;; Number of viceroys that have died.
    vcry-death          ;; Number of viceroys that have died
                        ;;    in that turn.
]

;;   This routine
;;   rounds the variable color to be the closest number
;;   ending in five.  The number five is chosen because
;;   colors ending in five are brightest and therefore
;;   easiest to see.
to-report get-fixed-color [col]
    report (((round (col / 10)) * 10) - 5)
end

to setup
  ca
  setup-variables
  setup-turtles
end

to setup-variables
  set ticks 0
  set carry-cap-mnrchs 175
  set carry-cap-vcrys 175
  set carry-cap-birds 50
  set reproduction-chance 4
  set color-range-begin 15
  set color-range-end 109
end

;; We create 50 birds and 350 butterflies of which half are
;;   monarchs and half are viceroys.  Initially, the
;;   monarchs are at the bottom of the color range and
;;   the viceroys are at the top of the color range.
;;   The screen is white for easy viewing.

to setup-turtles
  ask patches [ set pcolor white ]
  create-custom-birds carry-cap-birds
  [
    set color black
    setxy (random-float screen-size-x)
          (random-float screen-size-y)
    set memorycount1 -1
    set memorycount2 -1
    set memorycount3 -1

    ifelse (random 2) = 0 [
      set shape "bird 1"
    ][
      set shape "bird 2"
    ]
  ]

  create-custom-butterflies (carry-cap-mnrchs + carry-cap-vcrys)
  [
    ifelse (random 2) = 0 [
      set monarch? true
      set shape "butterfly monarch"
      set color (get-fixed-color color-range-begin)
    ][
      set monarch? false
      set shape "butterfly viceroy"
      set color (get-fixed-color color-range-end)
    ]
    setxy (random-float screen-size-x)
          (random-float screen-size-y)
  ]

end

to go
  ask birds [birds-move]
  ask butterflies [butterflies-move]
  ask butterflies [butterflies-get-eaten]
  ask birds [birds-forget]
  ask butterflies [butterflies-reproduce]

  update-data
  update-plot
  set ticks ticks + 1
  set m-death 0
  set v-death 0
end

;; The birds are sort of animated by alternating shapes.

to birds-move ;; birds procedure
  ifelse (shape = "bird 1") [
    set shape "bird 2"
  ][
    set shape "bird 1"
  ]

  set heading 180 + random-float 180.0
  fd 1
end

;; The butterflies move around randomly and identify a bird on their patch.
;;   If no birds are present the bird-here variable holds "nobody".

to butterflies-move ;; butterflies procedure
  rt random-float 100 - random-float 100
  fd 1
  set bird-here (one-of birds-here)
end

;; If a bird is present, the butterfly checks that bird's
;;   memory.  If the butterfly's color is not in the
;;   bird's yucky-color memory, the butterfly dies (is
;;   eaten).  If the butterfly is a monarch, it changes
;;   the bird's memory so that the bird remembers its
;;   color.

to butterflies-get-eaten ;; butterflies procedure
  if bird-here != nobody [
    ifelse ((color = memory1-of bird-here) or
           (color = memory2-of bird-here) or
           (color = memory3-of bird-here)) [
      stop
    ][
      ifelse monarch? [
        butterflies-teach-birds
        set m-death (m-death + 1)
      ][
        set v-death (v-death + 1)
      ]

      die
    ]
  ]
end


;; This routine is a complicated series of checks so that
;;   the new information about a yucky monarch color will
;;   replace the oldest information the bird has.  The
;;   bird can only remember three colors.  If a fourth
;;   color comes along, the bird has to forget one of the
;;   first three colors.  It forgets the color it learned
;;   first.

to butterflies-teach-birds ;; butterflies procedure
    if bird-here = nobody [ stop ]
    ifelse ((memory1-of bird-here) = 0) [
        set memory1-of bird-here color
        set memorycount1-of bird-here 0
    ][
        ifelse ((memory2-of bird-here) = 0) [
            set memory2-of bird-here color
            set memorycount2-of bird-here 0
        ][
            ifelse ((memory3-of bird-here) = 0) [
                set memory3-of bird-here color
                set memorycount3-of bird-here 0
            ][
                ifelse ((memorycount1-of bird-here) >
                               memorycount2-of bird-here) [
                    ifelse ((memorycount1-of bird-here) >
                               memorycount3-of bird-here) [
                        set memory1-of bird-here color
                        set memorycount1-of bird-here 0
                    ][
                        set memory3-of bird-here color
                        set memorycount3-of bird-here 0
                    ]
                ][
                    ifelse ((memorycount2-of bird-here) >
                               memorycount3-of bird-here) [
                        set memory2-of bird-here color
                        set memorycount2-of bird-here 0
                    ][
                        set memory3-of bird-here color
                        set memorycount3-of bird-here 0
                    ]
                ]
            ]
        ]
    ]
end

;; Birds can only remember a yucky color for so long, then
;;   they forget.  How long they can remember is
;;   determined by the slider MEMORY.

to birds-forget ;; birds procedure
  if (memorycount1 > -1) [
    set memorycount1 (memorycount1 + 1)
  ]
  if (memorycount2 > -1) [
    set memorycount2 (memorycount2 + 1)
  ]
  if (memorycount3 > -1) [
    set memorycount3 (memorycount3 + 1)
  ]

  if (memorycount1 > memory) [
    set memorycount1 -1
    set memory1 0
  ]
  if memorycount2 > memory [
    set memorycount2 -1
    set memory2 0
  ]
  if memorycount3 > memory [
    set memorycount3 -1
    set memory3 0
  ]
end

;; Each butterfly has an equal chance of reproducing
;;   depending on how close to carrying capacity the
;;   population is.  The chance that the butterfly will
;;   have a random color is determined by the MUTATION
;;   slider.

to butterflies-reproduce ;; butterflies procedure
    ifelse monarch? [
        if ((random mnrchs) < (carry-cap-mnrchs - mnrchs)) [
            if ((random-float 100) < reproduction-chance) [
                hatch 1 [
                    set monarch? true   ;; this line shouldn't be necessary -- bug in NetLogo
                    set heading (random-float 360)
                    fd 1
                    if ((random-float 100) < mutation-rate) [
                        set color (get-fixed-color ((random (color-range-end - color-range-begin)) + color-range-begin))
                    ]
                ]
            ]
        ]
    ][
        if ((random vcrys) < (carry-cap-vcrys - vcrys)) [
            if ((random-float 100) < reproduction-chance) [
                hatch 1 [
                    set monarch? false  ;; this line shouldn't be necessary -- bug in NetLogo
                    set heading (random-float 360)
                    fd 1
                    if ((random-float 100) < mutation-rate) [
                        set color (get-fixed-color ((random (color-range-end - color-range-begin)) + color-range-begin))
                    ]
                ]
            ]
        ]
    ]
end

;; This routine updates all the monitors.

to update-data
  set mnrchs
     (count butterflies with [monarch?])
  set mnrch-max
     (max values-from butterflies with [monarch?] [color])
  set mnrch-min
     (min values-from butterflies with [monarch?] [color])
  set mnrch-avg
     ((sum values-from butterflies with [monarch?] [color]) / mnrchs)
  set mnrch-death m-death
  set vcrys
     (count butterflies with [not monarch?])
  set vcry-max
     (max values-from butterflies with [not monarch?] [color])
  set vcry-min
     (min values-from butterflies with [not monarch?] [color])
  set vcry-avg
     ((sum values-from butterflies with [not monarch?] [color]) / vcrys)
  set vcry-death v-death
end

;; A straight-forward plot updating routine.

to update-plot
  set-current-plot-pen "Monarchs"
  plot mnrch-avg
  set-current-plot-pen "Viceroys"
  plot vcry-avg
end


; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the project: CONNECTED MATHEMATICS:
; MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL
; MODELS (OBPML).  The project gratefully acknowledges the support of the
; National Science Foundation (Applications of Advanced Technologies
; Program) -- grant numbers RED #9552950 and REC #9632612.
;
; Copyright 1998 by Uri Wilensky. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; This model was converted to NetLogo as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS.  The project gratefully acknowledges the support of the
; National Science Foundation (REPP program) -- grant number REC #9814682.
; Converted from StarLogoT to NetLogo, 2001.  Updated 2002.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (1998).  NetLogo Mimicry model.
; http://ccl.northwestern.edu/netlogo/models/Mimicry.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/Mimicry
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
250
10
632
413
15
15
12.0
1
10
1
1
1
0

CC-WINDOW
5
487
921
582
Command Center

BUTTON
6
49
61
82
setup
setup
NIL
1
T
OBSERVER
T
NIL

BUTTON
6
94
61
127
go
go
T
1
T
OBSERVER
T
NIL

SLIDER
79
48
243
81
memory
memory
0
40
30
1
1
NIL

SLIDER
79
92
243
125
mutation-rate
mutation-rate
0.0
100.0
10.0
1.0
1
NIL

MONITOR
5
177
99
226
Number Alive
mnrchs
0
1

MONITOR
6
323
98
372
Maximum Color
mnrch-max
3
1

MONITOR
6
374
99
423
Minimum Color
mnrch-min
3
1

MONITOR
6
424
99
473
Average Color
mnrch-avg
3
1

MONITOR
5
229
98
278
Number Dead
mnrch-death
3
1

TEXTBOX
6
156
134
177
Monarch Pops.

TEXTBOX
8
299
111
320
Monarch Colors

TEXTBOX
140
155
243
177
Viceroy Pops.

MONITOR
139
177
234
226
Number Alive
vcrys
0
1

MONITOR
139
228
234
277
Number Dead
vcry-death
0
1

TEXTBOX
143
299
246
325
Viceroy Colors

MONITOR
138
324
235
373
Maximum Color
vcry-max
3
1

MONITOR
138
374
235
423
Minimum Color
vcry-min
3
1

MONITOR
138
424
236
473
Average Color
vcry-avg
3
1

MONITOR
641
343
731
392
Ticks
ticks
0
1

PLOT
635
49
912
316
Average Colors Over Time
Time
Average Color
0.0
100.0
0.0
105.0
true
true
PENS
"Monarchs" 1.0 0 -65536 true
"Viceroys" 1.0 0 -16776961 true

@#$#@#$#@
WHAT IS IT?
-----------
"Batesian mimicry" is the term used to describe an evolutionary relationship in which a harmless species [the mimic] has evolved so that it looks very similar to a completely different species that isn't harmless [the model].  A classic example of Batesian mimicry is the similar appearance of monarch butterflies and viceroy moths. Monarchs and viceroys are unrelated species that are both colored similarly - bright orange with black patterns. Their colorations are so similar, in fact, that the two species are virtually indistinguishable from one another.

The classic explanation for this phenomenon is that monarchs taste yucky.  Because monarchs eat milkweed, a plant full of toxins, they become essentially inedible to birds.  Researchers have documented birds vomiting within minutes of eating monarch butterflies.  The birds then remember the experience and avoid brightly colored orange butterfly/moth species.  Viceroys, although perfectly edible, avoid predation if they are colored bright orange because birds can't tell the difference.

Recent research now suggests that viceroys might also be unpalatable to bird predators, confusing this elegant explanation.  However, we have modeled the relationship anyway.  Batesian mimicry occurs in enough other situations [snakes, for example] that the explanation's general truth is unquestionable.  The monarch-viceroy story is so accessible - and historically relevant - that we believe it to be instructive even if its accuracy is now questioned.

This model simulates the evolution of monarchs and viceroys from distinguishable, differently colored species to indistinguishable mimics and models.  At the simulation's beginning there are 350 monarchs and viceroys distributed randomly across the screen.  The monarchs are all colored red, while the viceroys are all colored blue.  They are also distinguishable (to the human observer only) by their shape:  monarchs are represented by the letter "x" while viceroys are represented by the letter "o".  50 birds are also randomly distributed across the screen.

When the model runs, the birds and butterflies [for the remainder of this description "butterfly" will be used as a general term for monarchs and viceroys, even though viceroys are technically moths] move randomly across the screen.  When a bird encounters a butterfly it eats the butterfly, unless it has a memory that the butterfly's color is "yucky."  If a bird eats a monarch, it acquires a memory of the butterfly's color as yucky.

As butterflies are eaten, they regenerate through asexual reproduction. Each turn, every butterfly must pass two "tests" in order to reproduce.  The first test is based on how many butterflies of that species already exist on the screen. The carrying capacity of the screen for each species is 150.  The chances of reproducing are smaller the closer to 150 each population gets.  The second test is simply a random test to keep reproduction in check [set to a 4% chance in this model].  When a butterfly does reproduce it either creates an offspring identical to itself or it creates a mutant.  Mutant offspring are the same species but have a random color between blue and red, but ending in five [e.g. color equals 15, 25, 35, 45, 55, 65, 75, 85, 95, 105].  Both monarchs and Viceroys have equal opportunities to reproduce mutants.

Birds can remember up to three yucky colors at a time. If a bird has memories of three yucky colors and it eats a monarch with a new yucky color, the bird "forgets" its oldest memory and replaces it with the new one.  Birds also forget yucky colors after a certain amount of time.


HOW TO USE IT
-------------
Each turn is called a TICK in this model.

The MEMORY slider determines how long a bird can remember a color as being yucky.  At the slider's maximum, a bird will remember a yucky color for thirty ticks.  At the slider's minimum, the bird will remember a yucky color for zero ticks.  In other words, it will not have any memory.

The MUTATION-RATE slider determines the chances that a butterfly's offspring will be a mutant.  Setting the slider to 100 will make every offspring a mutant.  Setting the slider to 0 will make no offspring a mutant.

The SETUP button resets the graphics and plots and randomly distributes the monarchs (all red), viceroys (all blue), and birds.  The GO button starts the simulation.

Numerous monitors track the two butterfly populations and their colors. The monarch monitors are situated under the headings "Monarch Population" and "Monarch Colors," while the viceroy monitors are situated under the headings "Viceroy Population" and "Viceroy Colors" NUMBER ALIVE tracks the current population level, while NUMBER DEAD tracks how many have died in the current turn. MAXIMUM COLOR and MINIMUM COLOR track the maximum and minimum colors in the monarch and viceroy populations, while AVERAGE COLOR tracks the average color in both populations. [In this model, blue - or color value 105 - is the highest color possible, while red - or color value 15 - is the lowest color possible].

The plot shows the average color of the monarchs and the average color of the Viceroys plotted against time.


THINGS TO NOTICE
----------------
Initially, the birds don't have any memory, so both monarchs and viceroys are eaten equally. However, soon the birds "learn" that red is a yucky color and most of the monarchs are protected by this.  As a result, the monarch population makes a comeback toward carrying capacity while the viceroy population continues to decline.  Notice also that as reproduction begins to replace eaten butterflies, some of the replacements are mutants and therefore randomly colored.
As the simulation progresses, birds continue to eat mostly butterflies that aren't red.  Occasionally, of course, a bird "forgets" that red is yucky, but a forgetful bird is immediately reminded when it eats another red monarch.  For the unlucky monarch that did the reminding, being red was no advantage, but every other red butterfly is safe from that bird for a while longer.  Monarch (non-red) mutants are therefore apt to be eaten.  Notice that throughout the simulation the average color of monarchs continues to be very close to its original value of 15.  A few mutant monarchs are always being born with random colors, but they never become dominant, as they and their offspring have a slim chance for survival.

Meanwhile, as the simulation continues, viceroys continue to be eaten, but as enough time passes, the chances are good that some viceroys will give birth to red mutants.  These butterflies and their offspring are likely to survive longer because they resemble the red monarchs.  With a mutation rate of 5%, it is likely that their offspring will be red too.  Soon most of the viceroy population is red.  With its protected coloration, the viceroy population will return to carrying capacity.

THINGS TO TRY
-------------

If the MUTATION-RATE is high, advantageous color genes do not reproduce themselves.  Conversely, if MUTATION-RATE is too low, the chances of an advantageous mutant (red) viceroy being born are so slim that it may not happen enough, and the population may go extinct.  What is the most ideal setting for the MUTATION-RATE slider so that a stable state emerges most quickly in which there are red monarchs and viceroys co-existing on the screen?  Why?

If the MEMORY slider is set too low, birds are unable to remember that certain colors are yucky.  How low can the MEMORY slider be set so that a stable state of co-existing red monarchs and viceroys emerges?

If you set MUTATION-RATE to 100 and MEMORY to 0, you will soon have two completely randomly colored populations.  Once the average color of both species is about 55, return the sliders to MUTATION-RATE equals 16 and MEMORY equals 30 without resetting the model.  Does a stable mimicry state emerge?  What is the "safe" color?


EXTENDING THE MODEL
-------------------
One very simple extension to this model is to add a RANDOM-COLOR button.  This button would give every butterfly on the screen a random color.  The advantage of red would be gone, but some color [which could be red, or any other color] would eventually emerge as the advantageous color.  This models the evolutionary process from an earlier starting place, presumably when even monarchs had different colors.

It would be interesting to see what would happen if birds were made smarter than they are in this model.  A smart bird should probably continue to experiment with yucky colors a few times before being "convinced" that all butterflies of that color are indeed distasteful.

You could try to add variables that kept track of how many yucky individuals of the same color a bird ate.  Presumably if a bird has eaten several monarchs that are all the same color, it will be especially attentive to avoiding that color as compared to if it had just eaten one butterfly of that color.  Making changes of this nature would presumably make the proportion of models and mimics more in keeping with the predictions of theorists that there are generally more models than mimics.  In the current model, birds aren't smart enough to learn that most butterflies may be harmless in a given situation.

In a real world situation, the birds would also reproduce.  Young birds would not have the experiences necessary to know which colors to avoid.  Reproduction of birds, depending on how it happened and how often, might change the dynamics of this model considerably.

One could also refine the mutation-making procedures of the model so that a butterfly is more likely to reproduce a mutant that is only slightly differently colored than to reproduce a mutant that is completely differently colored.  In the current model, mutants' colors are simply random.


CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (1998).  NetLogo Mimicry model. http://ccl.northwestern.edu/netlogo/models/Mimicry. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1998 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/Mimicry for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bird 1
false
0
Polygon -7566196 true true 2 6 2 39 270 298 297 298 299 271 187 160 279 75 276 22 100 67 31 0

bird 2
false
0
Polygon -7566196 true true 2 4 33 4 298 270 298 298 272 298 155 184 117 289 61 295 61 105 0 43

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

butterfly monarch
false
15
Line -1 true 0 0 424 424
Line -1 true 299 1 -128 424

butterfly viceroy
false
15
Circle -1 false true 34 34 232

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
